
;--------------------------------------------------------------------
; Main Menu routines
;
; SUFLAG usage:
;		bit 0 = start up director
;		bit 1 = free blocks indicator
; 		bit 2 = display block info indicator
;		bit 3 = sort order indicator
;		bit 4 = display enter text indicator
; 		bit 5 = main rom control by REX indicator
;		bit 6 = load pending indicator
;		bit 7 = commit load indicator

; variable type - 00=ramimage, 01=romimage, 10=bootimage, 11=system
; max 16 of each type
; assumptions - you never start with more than 16 of one type of image



; support for main rom replacement
; use bit 5 of SUFLAG to indicate if it REX is providing main rom
; detect main rom status my checking for "enu" vs "ENU"
; if main rom is active, display "M" at character 9
; if primary is active, display "P" at character 9
; if secondary is active, display "S" at character 9
; do not allow switching if target is blank
; display entry name for 2,4, ignore entry names for 3/5
; assign names to 2 and 4.



; about JMPVECT
; use this pointer to redirect control to the appropriate menu
; STACK should also be reset, and should be valid for any of the menus


; Warning messages to re-install ROMS and user programs that use hooks
;	when new RAM is installed
; 	when revert to previous ram image

; note about saving RAM images and V4.5 compatibility

;	V4.5 RAM images
;	- hooks table still installed in the RAM image (bad)
;	- rom traces are still there also
;	V4.6 RAM images
;	- hooks table saved to lower ram and wiped from upper RAM
;	- rom traces are still there also


;	when we start 4.6 REX Mgr, we wipe the hooks table and store it in ram
;	we do not clear out the rom program, or reset upper memory to default
; 	when ram images are then stored in REX, they have no hooks table
;	but the hooks table is in ram still.

; 	when we restore, we remove all traces of the main rom from the ram image
;	we also need to wipe the hooks table to default also because V4.5 images 
;	still have the hooks installed.

;	V4.6 restored RAM images then, should have no hooks and no rom traces.

;	- must add a hooks sweep out to the processes that load stored ram images
;	- must add warning screen to same.

; note about ROMSEL and Main Rom Management
; 	- the point is
;		- to define the working main rom image
;		- to store the info in the active block
;	- always run REX MANAGER with Primary rom (default power up condition)
;	- we should detect the state of rom selection on start of menu
;	- maintain that throughout operation
;	- if we change the use, change ROMSEL and update the active block info, no switch
;	- detect and store ROMSEL in actblk_routines
;	- make sure that RESTORE_AB is compatible.


;--------------------------------------------------------------------
REXMGR:
main_menu:

; interrupts should be on in REXMGR
; enter with HKFLAG set based on incoming hook table status

	di	
	call	I_REX_SW

	call	I_SETPRIMARY
	call	GEN_1		; make sure primary rom is selected

	ei			; REX MANAGER runs with primary rom selected
				; active block data holds desired main rom selection.

	xra	a		; set sort to newest last, turn off block info display	
	sta	SUFLAG		; don't display enter text by default
	sta	TYPE		;  set type to 00  00=RAM 01=ROM 10=OS 11 = SYS


main_menu2:

#if real_hardware = 1
	call	syscheck		; make sure system is backed up
					; and check directory also
#endif

	call	test_rom_mode
	call	display_title

	call	scan_directory_structure	; detailed check of directory structure

main_menu3:

;menu_loop0:				; use this loop for file operations
;menu_loop1:				; this loop for type updates
;menu_loop2:				; this loop for cursor updates
;menu_loop3:				; use this loop if cursor not updated, or keyboard input invalid

menu_loop0:				; use this loop for file operations


;menu_loop1:				; this loop for type updates

	lxi	h,menu_loop0
	shld	JMPVECT			; store this location as the restart of menu

	call	clsmsg	

	lda	TYPE
	cpi	03h
	jz	sysmenu			; system type, special menu.
	
					; ROMSEL should be always set correctly

	call	cls_enter

	call	display_rom_mode	; indicate if REX is in control of the ROM

	call	quick_scan_directory
	call	get_freeblock_status	; includes scan directory structure


	call	read_active_block	; update active block info first
					; update ROMSEL
					; must be ahead of scan_for_BB

	call	scan_for_BB		; point FREBLK to a free block pair
					; tidy up directory and block assignments to suit
					; always return pointing to a BB pair, FREBLK set

;	jz	scan_not_blank_error	; handle an error condition

	call	quick_scan_directory	; rescan the table to populate the tables again.

	call	display_group_type
	call	display_free_blocks

menu_loop00:				; enter here for re-sorted list
	call	directory_sort		; entries sorted

	call	blank_dir

	mvi	a,16d
	sta	TEMP4

	call	display_directory_entries	; requires updated COUNT

menu_zero_cursor:
	xra	a
	sta	CURSOR				; zero out cursor locations

menu_loop2:					; this loop for cursor updates
						; highlight the menu entry that is under the cursor


	xra	a		; now, find block number for item under cursor
	dcr	a
	sta	CURBLK		; store FF

				; if count > cursor, mode = 0 save
				; if count = cursor, mode = 1 load
	call	get_mode	; return with zero bit set for mode = 0
	push	psw		; store zero bit

	jnz 	skip_entry_block_number		; if load mode, skip updating CURBLK


	call	get_block_cursor

	push	h
	call	get_entry_date
	call	print_date

	pop	h
	call	get_entry_time
	call	print_time

	jmp	key1


skip_entry_block_number:

	call	cls_date

					; stack holds status of mode calculation
					; pop/push should allow jumps on these flags
					; zero set = mode 0 = save mode
					; zero not set = mode 1 = load mode
key1:	call	RPRT00
#if modeltype = 0			; M100
	.db	08d,01d
#else					; T200
	.db	012d,01d
#endif

	.db	"Help "
	.db	00

key2:	pop	psw		; restore zero bit
	jz	key2a		; jump to display save if mode = 0

	call	RPRT00		; display LOAD, erase keys 3-6
#if modeltype = 0			; M100
	.db	08d,06d
#else					; T200
	.db	012d,06d
#endif

	.db	"Load                     "
	.db	00
	jmp	key7

key2a:	call	RPRT00		; display Save, Copy
#if modeltype = 0			; M100
	.db	08d,06d
#else					; T200
	.db	012d,06d
#endif
	.db	"Save Copy "
	.db	00

	call	is_active	; zero set if active
	jz	key4a	

key4:	call	RPRT00		; display Kill
#if modeltype = 0			; M100
	.db	08d,16d
#else					; T200
	.db	012d,16d
#endif
	.db	"Kill "
	.db	00
	jmp	key5

key4a:	call	RPRT00		; display Kill
#if modeltype = 0			; M100
	.db	08d,16d
#else					; T200
	.db	012d,16d
#endif
	.db	"     "
	.db	00


key5:	call	RPRT00		; display Name, reserved for future use
#if modeltype = 0			; M100
	.db	08d,21d
#else					; T200
	.db	012d,21d
#endif
	.db	"Name      "
	.db	00


key7:	call	RPRT00
#if modeltype = 0			; M100
	.db	08d,31d
#else					; T200
	.db	012d,31d
#endif
	.db	"DeIn Exit"		; de-install REX , Exit
	.db	00

				; now get keyboard response, take action.

menu_loop3:			; use this loop if cursor not updated, or keyboard input invalid


	lda	SUFLAG
	ani	00000100b		; test for zero in bit 2
	jz	menu_loop4
					; if bit = 1 then show block info
	call	display_which_free_block
	call	display_current_block

menu_loop4:

	call	draw_cursor	; store entire state
				; toggle cursor 

	lda	SUFLAG
	ani	00010000b		; test bit 4
	jz 	menu_get_key

	call	display_enter_text	; if bit = 1 then show enter text


menu_get_key:


	rst	6
	.dw	KYREAD		; wait for a character, returned in a
				; carry set - function key pressed
				; zero set - no key found
	jz	menu_get_key

	call	draw_cursor	; store entire state
				; toggle cursor 
				
				; a key has been pressed 
				; arrow keys move cursor around
				; space advances cursor
				; enter "selects", action depends on cursor location
				; function keys take action depending on cursor location

				; check for function keys first

	jc	menu_function_key_pressed		; if carry set then A holds fun key number
				; some other key pressed
				; up is 30, down is 31, left is 29, right is 28
				; space is 32, enter is 13
				; tab is 9

				; s or S for sort order

	rst	6
	.dw	UPCAS		; convert to uppercase

	cpi	'S'
	jz	toggle_sort
	
	cpi	'B'
	cz	toggle_showblock

	cpi	'E'
	cz	toggle_showenter


	cpi	'I'		; display info about REX model etc.
	jz	show_REX_info

	cpi	'C'
	jz	display_checksum

	cpi	09d
	jz	increment_type

	cpi	13d
	jz	cursor_select

	call	process_cursor_type
; if CURSOR unchanged then menuloop3 else menuloop2

	lda	CURSOR
	cmp	h
	
	jnz	menu_loop2
	jmp	menu_loop3

;--------------------------------------------------------------------
process_cursor_type:	; enter with a holding keyboard input
			; c = COUNT

	lxi	h, COUNT
	mov	c,m		; c holds count

	push	psw
	lda	CURSOR
	mov	h,a		; load h with CURSOR value
	pop 	psw

	cpi	31d
	jz	cursor_down
	cpi	30d
	jz	cursor_up
	cpi	29d
	jz	cursor_left
	cpi	28d
	jz	cursor_right
	cpi	32d
	jz	cursor_right

	
	ret  			; cursor unchanged, h = cursor
				; menu_loop3 is the jump location

	
cursor_down:
	lda	CURSOR
	adi	03d
	cmp	c
	rp				; if cursor+3 >= count then can't use down

	sui	03d	
	cpi	12d			; compare to 12
	rp				; if greater, can't use down arrow

	adi	04d			; otherwise add 4 to cursor location
	sta	CURSOR			; otherwise update cursor location
	ret 				; loop back

cursor_up:
	lda	CURSOR			; load a with cursor location
	cpi	04h			; test against 4
	rm				; if less, can't use up arrow
	sui	04d			; otherwise update cursor
	sta	CURSOR
	ret 

cursor_right:
	lda	CURSOR
	cpi	15d				; compare to 15
	jp	cursor_right_zero		; if > = 15 then 0 and exit

	cmp	c				; compare to count
	jm	cursor_right_increment		; if < count then increment and exit
						; otherwise zero and exit
cursor_right_zero:				; zero
	xra	a
	jmp	cursor_right_exit
cursor_right_increment:
	inr 	a
cursor_right_exit:
	sta	CURSOR			; otherwise update cursor
	ret 

cursor_left:
	lda	CURSOR
	cpi	1d			; compare to 1
	jp	cursor_left2		; if >= 1 then decrement and exit

cursor_left1:				; wrap to max
	mov	a,c			; load a with count (0-16)
	cpi	16d
	jm	cursor_left_exit	; if < 16 store and exit
					
cursor_left2:				; decrement a and exit
	dcr	a			

cursor_left_exit:
	sta	CURSOR
	ret 	


;-------------------------------------------------------------------
increment_type:				
	lda	TYPE			; 00=RAM 01=ROM 10=OS 11 = SYS

increment_type_again:
	inr	a			; increment it
	ani	00000011b		; and with 00000011b (wrap it)

					; now, see if this is a valid result
	cpi	00000001b		; compare to 01, ROM type
	jz	increment_type_ok	; if valid type, jump out

#if no_SYS = 1
	cpi	00000011b		; compare to 011, SYS type
	jz	increment_type_ok	; if valid type, jump out
#endif
#if no_OS = 1
	cpi	00000010b		; compare to 010, OS type
	jz	increment_type_ok	; if valid type, jump out
#endif
#if no_RAM = 1
	cpi	00000000b		; compare to 000, RAM type
	jz	increment_type_ok	; if valid type, jump out
#endif
	jmp	increment_type_again

increment_type_ok:
	sta	TYPE
	lhld	JMPVECT
	pchl				; return to appropriate top menu
					; stack does not need resetting


;-----------------------------------------------------------------------------
get_freeblock_status:

;	call	scan_directory_structure	; b holds count of (A)
					; COUNT (A of TYPE) updated
	lda	SUFLAG
	ani	11111101b		; turn off bit 2
	sta	SUFLAG			; indicate if there is space for a new image

	mvi	a,30d
	sub	b			; if less than 2 free, we have a problem.
	mov	e,a			; e holds free blocks count
	sta	FREE			; update free blocks counter 

	push	psw
	cz	no_free_blocks_setflag
	pop	psw
	cm	no_free_blocks_setflag	; set SUFLAG = 1 - no space

	ret


;-------------------------------------------------------------------
menu_function_key_pressed:
; evaluate the function key action requested, based on mode.
; entry with a = function key number

	ora	a		; fn key 1
	jz	display_help

	push	psw		; temp store fn key result

	push	b
	call	get_mode
	pop	b
				; psw holds result of calculation (mode)
	jz	menu_fn_key_mode0

menu_fn_key_mode1:		; LOAD
	pop	psw		; restore fn key result

	dcr	a		; fn key 2
	jz	Load_image_mainmenu	
	
	dcr	a		;3
	dcr	a		;4
	dcr	a		;5
	dcr	a		;6
	
	jmp	menu_fn_key7

menu_fn_key_mode0:		: SAVE COPY KILL NAME
	pop	psw		; restore fn key result

	dcr	a		; fn key 2
	jz	Save_image	
	
	dcr	a		; 3
	jz	Copy_image

	mov	b,a
	call	is_active
	mov	a,b
	jz	skip_kill

	dcr	a		; 4
	jz	Kill_image
	inr	a

skip_kill:
	dcr	a
	dcr	a		; 5
	jz	Name_image

	dcr	a		; 6
;	jz	display_checksum		; reserve for future use
	

menu_fn_key7:
	dcr	a		; 7
	jz	de_install_REX

	dcr	a		; 8
	jz	quit			; quit with a hooks table rebuld
					; HKFLAG controls hook table rebuild
					; unchanged at this point

	jmp	menu_loop3		; must have pressed some other key, so loop back
	


;-------------------------------------------------------------------
display_type_text:
	.db	"RAM",00d	; 00
	.db	"ROM",00d	; 01
	.db	"OS ",00d	; 10
	.db	"SYS",00d	; 11

;---------------------------------------------------------------------------------

display_get_position:			; get position from count (reg c) for display
					; return row column in hl	
	
	mov	a,c			; get count, 0-15
					; each line is 4*6 + 3*4, with 2 spaces on each side
					; SSXXXXXXYSSSXXXXXXYSSSXXXXXXYSSSXXXXXXYS
					; row 3-6, column 3,13,23,33
					; entry is 6 characters + 1 character
					; 4 lines total = 16 entries

	ani	00001100b		; line indicator in bits 2 and 3
	rrc
	rrc
#if modeltype = 0			; M100
	adi	03d			; add 3 to offset to 3rd line (range 3-6)
#else					; T200
	adi	07d			; add 7 to offset to 3rd line (range 7-11)
#endif	

	mov	l,a			; store row number in l

	mov	a,c
	ani	00000011b		; column number 0-3

	mov	h,a
	mvi	a,249d
display_get_position1:
	adi	10d
	dcr	h
	jp	display_get_position1
					; 3,13,23,33
	mov	h,a			; hl = column, row - matches posit
	ret




;--------------------------------------------------------------------
; copy block in a to FREBLK
; updates directory
;--------------------------------------------------------------------
Copy_image:
cpyblk:
	call	check_flash_ready
	call	check_free_block		; check if there is space
	lda	FREBLK
	call	check_blank
	call	check_count

	lxi	h, msg01
	call	RPBMSG

	ori	01h		; reset zero bit
	lda	CURBLK
	call	copy_block	; copy from CURBLK to FREBLK

	jmp 	menu_loop0	; loop back

;-------------------------------------------------------------------
; copy_block subroutine
; copy from block in a to FREBLK, c = FREBLK
; make new directory entry as well.

; entry with zero flag set = preserve date/time, not set, new date/time

copy_block:

	push	psw			; temp store a

	call	clsbar

	call	I_BARS
	call	GEN_1			; set up bar indicator

	call	I_CPYBLK

#if real_hardware = 1

	pop	psw
	push	psw			; reload a
	mov	b,a			; set source
	lda	FREBLK
	mov	c,a			; set target
	push	d
	lxi	d, 08000h		; de = end start
	call	GEN_1			; copy from b to c
	pop	d
#endif
	
	call	is_name_directory_full
	cz	refresh_directories

	pop	psw			; reload a
	push	psw			; store flags
	call	find_directory_entryA	; get the address of the current block directory entry
	push	h			; store directory pointer on stack
	call	read_directory_lastb	; update lastb

	pop	h
	inx	h
	pop	psw
	jz	copy_freblk_entry	; do this one if zero set on entry
	jmp	write_freblk_entry	; do this one if not zero on entry




;--------------------------------------------------------------------
; erase block 
; enter with CURBLK = block to erase
; if not ready, display error message and return
; this routine no longer "kills" the block, it just marks the directory entry
;--------------------------------------------------------------------

erblk:	
Kill_image:				; enter with CURBLK = block to erase
					; can't kill active blocks



	lxi	h,msge5
	call	RPBMSG

	call	get_YN
	jnz	menu_loop2		; return to menu loop if not Y

	call	check_flash_ready

	lxi	h,msg03
	call	RPBMSG

	lda	CURBLK
	call	find_directory_entryA
	call	kill_directory_entry		; makes an A entry a U entry
	

	jmp	menu_loop0		; done


	


;-----------------------------------------------------------------------
; name the current block
; CURBLK holds block to be named or renamed

Name_image:
namblk:					; name the current block
					; find name in directory, and erase if present
					; get the new name
					; write to directory


	lxi	h,msg04
	call	RPBMSG


#if modeltype = 0			; M100
	lxi	h,0C08h
#else					; T200
	lxi	h,0C0Ch
#endif
	call	get_6_byte_name
					; entry with hl pointing to start of 6 byte name location on screen, formatted for POSIT
					; store 6 byte name at INPUT_BUFFER, F685 is first character of name
					; if escape hit then set zero bit
					; if successful, reset zero bit
					; enter with hl = col,row

	jz	menu_loop2		; if escape hit, then bail out (no cursor change)

	call	check_flash_ready

	lda	CURBLK
	call	find_directory_entryA
	push 	h

	lxi	d,07h
	dad	d
	mov	a,m	
	sta	LASTB			; store last byte

	pop	h
	push	h			; store pointer to original entry
	call	blank_directory_entry	; blank this directory entry
		
	call	is_name_directory_full
	cz	refresh_directories	; if end of record then we must rebuild the directories

	mvi	a,0FFh
	call	find_directory_entry	; get hl = open slot location

	push	h			; h = location for new record
	lda	CURBLK
	call	write_directory_block	; entry 	hl = pointer to directory entry, a = block number
					; exit 		none, block written

	lxi	h,INPUT_BUFFER		; get pointer to 6 byte name		
	xchg				; d points to first character

	pop	h			; hl holds new record location, de holds pointer to text		
	push	h
	call	write_directory_name	; write name bytes

	pop	h
	push	h
	call	write_directory_lastb	; write lastb

	pop	h			; hl = new record location
	pop	d			; de = old record location

	lxi	b,08d
	dad	b
	xchg
	dad	b
	xchg				; hl and de advanced 8 bytes

	mvi	b,04d

name_data_loop:				; copy 4 bytes of date and time
	ldax	d			; load a with source data
	push	b
	push	d
	call	double_byte_poke	; store in new record
	pop	d
	pop	b
	inx	h
	inx	d			; advance de, hl
	dcr	b			; count down
	jnz	name_data_loop
	

	call	fix_input_buffer

	jmp	menu_loop0		; return to menu loop
	

;----------------------------------------------------------------------
display_checksum:			; calculate curblk checksum and display

	lda	CURBLK
	cpi	0ffh
	jz	menu_loop2

	call	clsmsg

	lxi	h,msg10
	call	RPBMSG			; display message



	call	I_CHKSUM

	lda	CURBLK

	lxi	d, 08000h		; de = end start
	call	GEN_1			; calculate checksum in block a

	mov 	a,d
	call	print_a_as_2_hex_char
	mov	a,e	
	call	print_a_as_2_hex_char

	lxi	h,msg11
	call	RPBUF

	call	hit_any_key

	lhld	JMPVECT
	pchl
;	jmp	menu_loop0




;--------------------------------------------------------------------

cursor_select:
evaluate_enter_key:
; type RAM
; enter (actram) - store RAM to actram, return to menu
; enter (not actram) - backup RAM to actram, install RAM from entry, new ACTRAM, exit
; enter (blank)	- backup RAM to new block, ACTRAM=NEWBLOCK, return to menu
; type ROM and OS
; enter (actopt) - do nothing (new!)
; enter (not actopt) - install new ROM and start new ROM
; enter (blank) - do nothing

	lda	TYPE
	cpi	00h		 	;00=ramimage, 01=romimage, 10=bootimage, 11=system
	jz	enter_type_is_RAM
	cpi	03h
	jz	menu_loop2		; loop back if system type
;--------------------------------------------------------------------------	
enter_type_is_ROM_OS:			; figure out what to do
	
	lda	CURBLK
	cpi	0FFh
	jz	menu_loop2		; return if blank - do nothing

	lda	CURBLK
	mov	b,a
	lda	ACTOPT
	cmp	b
	jz	reactivate_rom		; if CURBLK = ACTOPT then reactivate

					; must be a valid rom
	lxi	h,msge4
	call	RPBMSG

	call	get_YN
	jnz	menu_loop2		; return to menu loop if not Y
	
	rst	6
	.dw	LCD			; print 'Y' at cursor

	lda	CURBLK
	sta	ACTOPT			; if not, set new ACTOPT
	
	call	update_actblk_directory	; make new directory entry


install_new_ROM:			; jump to here to remove existing rom and install new


	lda	HKFLAG			; get HKFLAG
	ora	a			; test for 1
	cnz	display_warning_rom	; if 1, must warn that hooks initialized

					; hook table already initialized, no restore of hooks
	call	remove_ROM		; remove traces of ROM from RAM

	jmp	activate_rom

;--------------------------------------------------------------------------
reactivate_rom:

	di
#if storehooks = 1 		
	lda	HKFLAG
	ora	a
	cnz	restore_hooks_table	; test HKFLAG for 1, if 1 then restore hooks
#endif


activate_rom:			; no need for warning on hooks restore, same optrom
	xra	a
	sta 	SUFLAG			; no particular reason
	sta	HKFLAG
	call	fix_input_buffer
	call	find_rom_call
	push 	h
	push 	psw
	di
#if install_hook = 1
	call	install_tmrhk	
#endif
					; install same rom, and exit to ROM
	call	I_ROMST
	pop	psw
	pop	h
	jmp	GEN_1		; we don't return from this--

;--------------------------------------------------------------------------
enter_type_is_RAM:			; figure out what to do

	lda	CURBLK
	cpi	0FFh			; compare CURBLK to FF
	jz	enter_RAM_blank		; if blank, then store fresh image

	mov	b,a
	lda	ACTRAM			; compare CURBLK to ACTRAM
	cmp	b
	jz	enter_actram		; refresh this


;--------------------------------------------------------------------------
enter_not_actram:			; backup RAM to ACTRAM
					; install RAM from CURBLK
					; ACTRAM=CURBLK
					; exit
					; "switching and exiting"
					; delete old ROM
					; (removed) install new ROM
	lxi	h,msge1
	call	RPBMSG

	call	get_YN
	jnz	menu_loop2		; return to menu loop if not Y

	rst	6
	.dw	LCD			; print 'Y' at cursor

	call	refresh_backup		; make new ram backup
					; copy name entry to new entry for FREBLK
					; kill ACTRAM entry

	lda	FREBLK
	sta	ACTRAM

	call	update_actblk_directory
	
	lda	CURBLK
	call	FLASH_to_RAM		; install new RAM

	lda	CURBLK
	sta	ACTRAM			; new ACTRAM
	
	call	update_actblk_directory

;--------------------------------------------------------------------------
new_ram_loaded:

#if modeltype=1			; T200
	lxi	h, 0EEF5h	; point to memory location for active block
	mvi	m,00h		; set to 00h
				; if bank1 active, then this is correct
				; if bank2 or 3 active, then this is a don't care.
#endif


	lda	HKFLAG
	ora	a			; test HKFLAG, if 1 then we must warn

	cnz	display_warning_ram	; if not zero, display warning message.

	call	remove_ROM		; remove traces of ROM from RAM

	xra	a
	sta	HKFLAG			; new ram - no hooks restore
	
;	jmp	quit			; restart 

; must exit through REX_INSTALL in order to purge old REXMGR files.
	
	mvi	a,0FFh
	sta	SUFLAG

	jmp	REX_RAM_INSTALL
	

;--------------------------------------------------------------------------
enter_actram:				; backup RAM to ACTRAM
					; return to menu
					; refrest or revert

enter_actram_refresh:		
	lxi	h,msge2
	call	RPBMSG

	call	get_YN
	jnz	enter_actram_revert	; revert?

	rst	6			; execute refresh
	.dw	LCD			; print 'Y' at cursor

	call	refresh_backup		; make new ram backup
					; copy name entry to new entry for FREBLK
					; kill ACTRAM entry

	lda	FREBLK
	sta	ACTRAM

	call	update_actblk_directory

	jmp	menu_loop0


enter_actram_revert:
	lxi	h,msge6
	call	RPBMSG

	call	get_YN
	jnz	menu_loop2		; return to menu loop

	rst	6			; execute revert
	.dw	LCD			; print 'Y' at cursor


	lda	CURBLK
	call	FLASH_to_RAM		; install new RAM
	

	jmp	new_ram_loaded




;--------------------------------------------------------------------------

enter_RAM_blank:			; backup RAM to fresh entry
					; ACTRAM = new entry
					; get name for new entry
					; return to menu
					; "creating new backup..."

	call	check_free_block

	lxi	h,msge3
	call	RPBMSG

	call	get_YN
	jnz	menu_loop2		; return to menu loop if not

	rst	6
	.dw	LCD			; print 'Y' at cursor
	
	lxi	h,msg04
	call	RPBMSG

#if modeltype = 0			; M100
	lxi	h,0C08h
#else					; T200
	lxi	h,0C0Ch
#endif

	call	get_6_byte_name
					; entry with hl pointing to start of 6 byte name location on screen, formatted for POSIT
					; store 6 byte name at INPUT_BUFFER, F685 is first character of name
					; if escape hit then set zero bit
					; if successful, reset zero bit
					; enter with hl = col,row
	jz	menu_loop2		; if escape hit, then bail out (no cursor change)

	lxi	h,INPUT_BUFFER

	call	backup_ram		; copy RAM to FREBLK
					; move name entry to new entry for FREBLK
					; enter with hl pointing to name text

	lda	FREBLK
	sta	ACTRAM

	call	update_actblk_directory	; new routine to update the ACTBLK directory			

	call	fix_input_buffer
	jmp	menu_loop0	
	



;--------------------------------------------------------------------------
refresh_backup:				; make new ram backup
					; copy name entry to new entry for FREBLK
					; kill ACTRAM entry
					
	lda	ACTRAM
	call	find_directory_entryA
	push	h

	inx	h
	call	backup_ram		; copy RAM to FREBLK
					; move name entry to new entry for FREBLK
					; enter with hl pointing to name text
	pop	h
	call	kill_directory_entry	; mark existing as killed - mark old actram as U

	ret




;--------------------------------------------------------------------------
; message text

msge1:	.db	"Switch and exit (Y/N)? ",00h
msge2:	.db	"Refresh backup (Y/N)? ",00h
msge3:	.db	"Create new backup (Y/N)? ",00h
msge4:	.db	"Install ROM (Y/N) ? ", 00h
msge5:	.db	"Kill image (Y/N) ? ",00h
msge6:	.db	"Revert to backup (Y/N)? ",00h

msg01:	.db	"Copying image......",000h
msg02:	.db	"Rebuilding directories...",000h
msg03:	.db	"Killing image.......",000h
msg04:	.db	"New Name ? ",000h
	
msg08:	.db	"RETRY!",00h

msg10:	.db	"16 bit checksum = ",00h
msg11:	.db	"  <any key>..",00h

blank_entry_text:
	.db	"------ ",00h

areyousure:
	.db	"Are you sure? (Y/N) ", 00h

;-----------------------------------------------------
;normal_cursor:
;	push	b
;	lda	CURSOR				; first go normal video on old cursor
;	call	draw_cursor
;	pop	b
;	ret

;------------------------------------------------------
get_mode:			; return with zero bit set = mode 0
				; zero bit reset = mode 1

	lda	COUNT		; figure out if mode is 1 or 0
	mov	b,a		; count in b
	
	lda	CURSOR		; cursor in a
	cmp	b		; compare to count
	jnz	mode_is_zero

mode_is_one:
	mvi	a,01h
	ora	a		; reset zero bit
	ret		

mode_is_zero:
	xra	a		; set zero bit
	ret

;------------------------------------------------------
is_active:			; return with zero bit set = active
				; zero bit reset = not active

	push	b

	lda	CURBLK		; figure out if block is active
	mov	b,a		; count in b
	
	lda	ACTOPT		; cursor in a
	cmp	b		; compare to count
	jz	exit_here

	lda	ACTRAM		; cursor in a
	cmp	b		; compare to count

exit_here:
	pop	b
	ret


;-----------------------------------------------------------------
backup_ram:			; copy RAM to FREBLK
				; move name entry to new entry for FREBLK
				; enter with hl pointing to name text
	
	push	h			
	call	RAM_to_FLASH		; copy current RAM to FREBLK, refreshing it
					; no directory updates
	pop	h
	xra	a
	sta	LASTB
	call	write_freblk_entry	; enter with hl pointing to entry with name text
					; assumes LASTB is updated
	ret



;-------------------------------------------------------------------
msgtemp:	.text	"hit any key...test"
		.db	00h

scan_for_BB:				; find a BB pair 
					; garbage collection routine
					; assumes a valid directory scan is in place!
					; copies kill the directory scan....
					; return not zero if there is an error

scan_BB:	



	mvi	a,0FFH
	sta	FREBLK			; since looking for a new free block, reset FREBLK

	lxi	d,0000h
	call	search_sector	

	jnz	scan_UU			; jump to look for a free UU pair to erase
	
	mov	a,c
	ani	11111110b		; just to be safe, always pick even block number
	sta	FREBLK			; we found a blank pair

	; precautionary blank checks...
;	call	check_blank		; if not blank, then restart
;	rnz

;	lda	FREBLK
;	xri	00000001b
;	call	check_blank		; if not blank, then restart

;	call	quick_scan_directory

	ret				; success
					; zero set, ok, zero reset, error







scan_UU:
	lxi	d,4040h			; 64d, 64d
	call	search_sector
	jnz	scan_BU			; if not found look for BU
	jmp	scan_erase_repeat	; if found, erase and repeat

scan_BU:				; look for a BU combination
	lxi	d,0040h
	call	search_sector
	jnz	scan_XA			; if no single U then scan XA
	
scan_erase_repeat:			; if found, erase and repeat
	mov	a,c
	call	erase_pair
	jmp	search_sector_repeat	; repeat


scan_XA:				; there are no UU or BB pairs
					; there are no UB pairs
					; scan for single B
					; then look for a single A in a different block

	lxi	d,0C000h		; look for a BA pair
	call	search_sector

	jnz	search_sector_error	; if we don't find at least one single, problem!

					; found a B			
	mov	a,c			; c points B
	sta	FREBLK
					; now look for a UA or BA
	lxi	d,040C0h		; look for AU
	call	search_sector

	jz	search_XA_foundA

	lxi	d,000C0h			; now search for AB pair (excluding B found previously)
	call	search_sector

	jnz	search_sector_error


search_XA_foundA:			; we found an A to copy away
					; c holds source A

	xra	a			; set zero flag
	mov	a,c

	push	b
	call	copy_block		; copy block in a to FREBLK
	pop	b
	mov 	a,c
	push	b
	call	erase_pair		; erase block in a
	pop	b

scan_XA_test_ACTRAM:
	lda	ACTRAM
	cmp	c
	jnz	scan_XA_test_ACTOPT

	lda	FREBLK
	sta	ACTRAM			; correct ACTRAM
	call	update_actblk_directory

	jmp	search_sector_repeat	; repeat


scan_XA_test_ACTOPT:
	lda	ACTOPT
	cmp	c
	jnz	search_sector_repeat	; repeat

	lda	FREBLK
	sta	ACTOPT			; correct ACTOPT
	call	update_actblk_directory


search_sector_repeat:

	call	quick_scan_directory		; call this for a fast scan
	jmp	scan_for_BB



;----------------------------------------------------------------------------------
erase_pair:					; enter with a= block to erase
						; both must be U or B, not A

	push	psw
	call	erase_block_a			; erase block in a

erase_first:
	pop	psw
	push	psw
	call	find_directory_entry		; hl points to directory entry
; entry 	a = block number to be searched for
; exit 		hl = pointer to directory entry (or potentially end of record) (or free entry)
; 		zero set if found, zero not set if not found
	
	jnz	erase_second			; if not found try the other 
	call	blank_directory_entry

erase_second:
	pop	psw
	xri	00000001b
	call	find_directory_entry		; hl points to directory entry
	rnz					; return if not found
	call	blank_directory_entry		; blank the entries

	ret


;-----------------------------------------------------------------------------
search_sector_error:
	lxi	h,msgsse
	jmp 	no_free_blocks_common

msgsse:	.db	"Can't find a blank sector!",00d				 
	
;-----------------------------------------------------------------------------
no_free_blocks_error:
	lxi	h,msgnoB
	jmp	no_free_blocks_common
	
msgnoB:	.db	"No free blocks!",00d

;-----------------------------------------------------------------------------
no_free_blocks_common:
	call	RPBMSG
	call	DELAY

no_free_blocks_setflag:
	lda	SUFLAG
	ori	00000010b	; turn on bit 1 - no space available
	sta	SUFLAG
	ret			; loop back

;------------------------------------------------------------------------	
display_free_blocks:		; enter with FREE updated.

	call	RPRT00
#if modeltype = 0			; M100
	.db	01h,21h
#else					; T200
	.db	05h,21h
#endif

	.db	"Free:"
	.db	00h

	lda	FREE
	ora	a
	rm			; don't display if a negative number

	jmp	display_which_block

;------------------------------------------------------------------------
display_which_free_block:	

	call	RPRT00
#if modeltype = 0			; M100
	.db	07h,01d
#else					; T200
	.db	0Bh,01d
#endif

	.db	"F:"
	.db	00h

	lda	FREBLK
	jmp	display_which_block

;------------------------------------------------------------------------
display_current_block:	

	call	RPRT00
#if modeltype = 0			; M100
	.db	07h,6d
#else					; T200
	.db	0Bh,6d
#endif

	.db	"C:"
	.db	00h

	lda	CURBLK

;------------------------------------------------------------------------
display_which_block:
	mov	l,a
	mvi	h,00h
	rst	6
	.dw	NUMPRT	

	mvi	a,' '
	rst	6
	.dw	LCD

	rst	6
	.dw	LCD

	ret

;------------------------------------------------------------------------
display_group_type:		
	call	RPRT00
#if modeltype = 0			; M100
	.db	01h,0Dh
#else					; T200
	.db	05h,0Dh
#endif

	.db	"Group<TAB>:"
	.db	00

#if modeltype = 0			; M100
	lxi	h,1901h
#else					; T200
	lxi	h,1905h
#endif

	rst	6
	.dw	POSIT

	lxi	h,display_type_text

	lda	TYPE
	rlc
	rlc
	mvi	d,00h
	mov	e,a
	dad	d
	
	call	RPBUF
	ret

;-------------------------------------------------------------------------------


display_directory_entries:
					; show all image entries of type in DISPLAY
					; start at beginning of table, display entries of specific type
					; for each one found that is of type, display, count
					; if <16 then last one is a ------ entry
					; display - lower 2 bits is type, next 2 bits is mode, upper 4 bits is count

					; table created at GEN_1+32d (64 bytes) with pointers to 
					; directory entries


	mvi	c,0FFh			; position counter in table
					; index to entry table at GEN_1
display_entries1:
	inr	c

	lda	COUNT
	cmp	c			; compare c to COUNT
	jz	display_entries_lastchar	; if equal then done

	lxi	h,GEN_1+96d

	push	b
	mvi	b,00h
	dad	b
	mov	a,m			; get block number to display
	pop	b

	lxi	h,GEN_1+32d
	push	b			; temp store bc
	add	a			
	mov	c,a			; c=a*2
	mvi	b,00h			; set b = 0
	dad	b			; hl = address of table entry
	pop	b			; restore bc

	xchg				; de = pointer to table entry 
	lhlx				; load hl with address of directory entry


	mov	a,m			; get block number
	ani	00111111b
	push	psw			; temp store block number

	inx	h			; hl points to start of name text
	push	b
	push	h
					; counter in c
					; position is 0-15
	call	display_get_position	; get position from count for display
					; return row column in hl
	rst	6
	.dw	POSIT
	pop	h
	pop	b


	mvi	d,06h
display_entries2:			; print the name
	mov	a,m
	rst	6
	.dw 	LCD			; no need to stack up for this routine



	inx	h
	dcr	d
	jnz	display_entries2


	pop	psw			; get back block number
	mov	e,a			; place in e

	lda	ACTRAM
	cmp	e
	jz	display_star		; display a * if entry = ACTRAM

	lda	ACTOPT
	cmp	e	
	jnz	display_entries1

display_star:				; display a * if entry = ACTOPT
	mvi	a,'*'
	rst	6
	.dw	LCD
	jmp	display_entries1


display_entries_lastchar:
	push	b
	lda	TEMP4
	mov	b,a
	lda	COUNT
	cmp	b
	pop	b
	rp				; if >= 16 then done (TEMP4)

	mov	c,a
	call  	display_get_position	; set hl for POSIT based on value of c
	rst	6
	.dw	POSIT			; set the cursor location

	lxi	h, blank_entry_text	; print ------
	call	RPBUF
	
	ret

;--------------------------------------------------------------------------
	
display_title:
	rst	6
	.dw	CLS

	call	RPRT00
#if modeltype = 0			; M100
	.db	01h,01h
#else					; T200
	.db	05h,01h
#endif

	.db	"REX",version
	.db	00h
	ret

;--------------------------------------------------------------------------

display_enter_text:		; entry with type valid, and CURBLK updated
				; types 00=RAM 01=ROM 10=OS 11 = SYS


	lda	TYPE
	ora	a
	jz	display_enter_text_ram

	cpi	03h
	jz	cls_enter			; return if type = sys

display_enter_text_rom_os:

	lda	CURBLK
	inr	a
	jz	cls_enter			; if CURBLK = FF, then clear line 2

	push 	psw
	call	display_enter_line
	pop	psw

	dcr	a
	mov	b,a
	lda	ACTOPT
	cmp	b
	jz	display_enter_text_actopt	; if CURBLK = ACTOPT, then clear line 2


	call	RPRT00
#if modeltype = 0			; M100
	.db	02d,10d
#else					; T200
	.db	06d,10d
#endif

	.db	"Switch OPTROM image "
	.db	00h
	ret

display_enter_text_actopt:

	call	RPRT00
#if modeltype = 0			; M100
	.db	02d,10d
#else					; T200
	.db	06d,10d
#endif

	.db	"Reactivate OPTROM   "
	.db	00h
	ret


display_enter_text_ram:
	call	display_enter_line

	lda	CURBLK
	inr	a
	jz	display_enter_blank

	lda	ACTRAM
	mov	b,a
	lda	CURBLK
	cmp	b
	jz	display_enter_text_ram_active

		; block is not active - refresh and switch

	call	RPRT00
#if modeltype = 0			; M100
	.db	02d,10d
#else					; T200
	.db	06d,10d
#endif

	.db	"Refresh and switch "
	.db	00h
	ret

display_enter_text_ram_active:	; display when cursor on active block
	call	RPRT00
#if modeltype = 0			; M100
	.db	02d,10d
#else					; T200
	.db	06d,10d
#endif

	.db	"Refresh or revert   "
	.db	00h
	ret


display_enter_blank:		; display when cursor on a blank slot
	call	RPRT00
#if modeltype = 0			; M100
	.db	02d,10d
#else					; T200
	.db	06d,10d
#endif

	.db	"Create new backup   "
	.db	00h
	ret


display_enter_line:
	call	RPRT00
#if modeltype = 0			; M100
	.db	02h,01h
#else					; T200
	.db	06h,01h
#endif

	.db	"<ENTER>: "
	.db	00h

	ret


;-----------------------------------------------------------------------
toggle_sort:
	lda	SUFLAG
	xri	00001000b
	sta	SUFLAG		; toggle bit 3

	jmp	menu_loop00

;-----------------------------------------------------------------------
toggle_showblock:
	push 	psw
	lda	SUFLAG
	xri	00000100b
	sta	SUFLAG		; toggle bit 2

	ani	00000100b
	push	psw
	cz	cls_blockinfo

	pop	psw
	push	psw
	cnz	display_which_free_block
	pop	psw
	cnz	display_current_block

	pop	psw
	ret

		

;-----------------------------------------------------------------------
toggle_showenter:
	push	psw
	lda	SUFLAG
	xri	00010000b
	sta	SUFLAG		; toggle bit 2

	ani	00010000b
	cz	cls_enter

	pop 	psw
	ret
; 	jmp	menu_loop3


;-----------------------------------------------------------------------
; start up SYSTEM to SYSBAK comparison
; compare directory block and sw block
; compare checksums

syscheck:

				; get checksums from directories in block 0 and 1 - compare directories
				; if not equal, choose which directory to keep
				;	- keep main directory
				; 	- keep backup dir - copy 1dir to 0dir
				; get checksums from bottom 32k in block 0 and 1
				; if different, copy 0 to 1


	call	RPRT00
#if modeltype = 0			; M100
	.db	02h,01h
#else					; T200
	.db	06h,01h
#endif

	.db	"Comparing directories ...."
	.db	00h

	call	I_CHKSUM


syscheck_testdir:

	xra	a
	lxi	d, 06040h	; de = end start
	call	GEN_1		; de holds 0sw checksum

	push	d		; temp store checksum
	
	mvi	a,01h
	lxi	d, 06040h	; de = end start
	call	GEN_1		; de holds 1sw checksum
	
	pop	h		; get hl from stack


	RST	6
	.dw	0018h		; compare the two checksums

	jnz	syscheck_diractions

	call	ok
	jmp	syscheck_32k


syscheck_diractions:		; choose which directory to keep
				;	- keep main directory - leave bit 1 = 1
				; 	- keep backup dir - copy 1dir to 0dir, set bit 1 = 0

	call	clsmsg 

	call	RPRT00
#if modeltype = 0			; M100
	.db	03h,01h
#else					; T200
	.db	07h,01h
#endif

	.db	"Directories are different."
	.db 	0FFh
#if modeltype = 0			; M100
	.db	07h,26d
#else					; T200
	.db	011h,26d
#endif

	.db	"   "
	.db	0FFh
#if modeltype = 0			; M100
	.db	07h,01h
#else					; T200
	.db	00Bh,01h
#endif

	.db	"Keep main directory? (Y/N) "
	.db	00h

	call	get_YN
	
	rst	6
	.dw	LCD		; print response
	
	mov	b,a		; store answer
	push	b

	lxi	h,areyousure
	call	RPBMSG

	call	get_YN

	push	psw
	push	b
	rst	6
	.dw	LCD
	pop	b
	pop	psw

	pop	b		; get back answer
	mov	a,b		; set answer in a

	jnz	syscheck_diractions	; if confirm Y, then answer valid, continue


	cpi	'Y'		; compare choice to if Y, then we want to keep current dir
	jz	syscheck_keepdir
				; answer was N, we are reverting to old directory
	call	clsmsg
	call	clsr7

	call	RPRT00
#if modeltype = 0			; M100
	.db	04h,01h
#else					; T200
	.db	08h,01h
#endif

	.db	"Copy backup directory to main ...."
	.db 	00h

	lxi	h,04000h
	xra	a
	call	erase_block	; erase directory block

	call	I_BARS
	call	GEN_1		; set up bar indicator

	call	I_CPYBLK

	lxi	d,06040h
	lxi	b,00100h	; b = 01, c = 00
	call	GEN_1		; copy block from b to c

	call	clsbar

	call	RPRT00
#if modeltype = 0			; M100
	.db	04h,34d
#else					; T200
	.db	08h,34d
#endif

	.db	" "
	.db 	00h

	call	ok
	jmp	syscheck_32k

syscheck_keepdir:

	call	clsmsg
	call	clsr7

	call	RPRT00
#if modeltype = 0			; M100
	.db	04h,01h
#else					; T200
	.db	08h,01h
#endif

	.db	"Keeping main directory."
	.db 	00h
	
				; directories are now equal
syscheck_32k:			; check entire block for equality


	call	RPRT00
#if modeltype = 0			; M100
	.db	05h,01h
#else					; T200
	.db	09h,01h
#endif

	.db	"Comparing SYSTEM to SYSBAK ...."
	.db 	00h
	call	I_CHKSUM

	xra	a
	lxi	d, 08000h	; de = end start
	call	GEN_1		; de holds 0sw checksum

	push	d		; temp store checksum
	
	mvi	a,01h
	lxi	d, 08000h	; de = end start
	call	GEN_1		; de holds 1sw checksum
	
	pop	h		; get hl from stack

	RST	6
	.dw	0018h		; compare the two checksums

	jz	syscheck_ok	; if equal, done, return

	call	RPRT00
#if modeltype = 0			; M100
	.db	06h,01h
#else					; T200
	.db	0Ah,01h
#endif

	.db	"Copy SYSTEM to SYSBAK ...."
	.db	00h

	mvi	a,01h
	sta	FREBLK
	call	erb0		; erase block 1

	call	I_BARS
	call	GEN_1		; set up bar indicator

	call	I_CPYBLK	; copy block from b to c

	lxi	b,0001h		; b = 00 c = 01
	lxi	d, 08000h	; de = end start
	call	GEN_1		; copy SYSTEM to SYSBAK

				; we did a system copy

	call	RPRT00
#if modeltype = 0			; M100
	.db	06h,27d
#else					; T200
	.db	0Ah,27d
#endif

	.db	" "
	.db	00h
	

syscheck_ok:
	call	ok
	ret


ok:	lxi	h,ok_text
	call	RPBUF
	ret
	

ok_text:	.db	" OK!",00h

;-----------------------------------------------------------------------
show_REX_info:			; display model, firmware codes
				; REX model = REX or REX2 (top 2 bits)
				; firmware 1 = number 0-15 (middle 4 bits)
				; firmware 2 = number 0-3 (bottom 2 bits)

	rst	6
	.dw	CLS

#if real_hardware = 1 
	call	I_STV
	call	GEN_1
					; STV
					; get status and version byte
					; firmware byte in c, status in b
#else
	mvi	c,01000111b
#endif

	push	b

	call	RPRT00
#if modeltype = 0			; M100
	.db	02h,02h
#else					; T200
	.db	06h,02h
#endif

	.db	"REX Software     :",version,", CS="
	.db	000h
	

	call	I_CHKSUM		; display system rom checksum
	xra	a

	lxi	d, 04000h		; de = end start
	call	GEN_1			; calculate checksum in block a

	mov 	a,d
	call	print_a_as_2_hex_char
	mov	a,e	
	call	print_a_as_2_hex_char

	call	RPRT00
#if modeltype = 0			; M100
	.db	03h,02h
#else					; T200
	.db	07h,02h
#endif

	.db	"REX Model        :"
	.db	00h

	pop	b
	push	b
	mov	a,c
	ani	11000000b
	
	jz	model_is_REX
	
	call	RPRT00
#if modeltype = 0			; M100
	.db	03h,20d
#else					; T200
	.db	07h,20d
#endif

	.db	"REX2"
	.db	00

	jmp	show_REX_info1

model_is_REX:
		
	call	RPRT00
#if modeltype = 0			; M100
	.db	03h,20d
#else					; T200
	.db	07h,20d
#endif

	.db	"REX"
	.db	00

show_REX_info1:

	call	RPRT00
#if modeltype = 0			; M100
	.db	04h,02h
#else					; T200
	.db	08h,02h
#endif

	.db	"REX Firmware     :"
	.db	00h

	pop	b
	push	b

	mov	a,c
	ani	00111100b

	rrc
	rrc

	mvi	h,00h
	mov	l,a
	
	rst	6
	.dw	NUMPRT			; print number in HL at cursor

	mvi	a,'('
	rst	6
	.dw 	LCD

	pop	b
	push	b

	mov	a,c
	ani	00000011b

	mvi	h,00h
	mov	l,a
	
	rst	6
	.dw	NUMPRT			; print number in HL at cursor

	mvi	a,')'
	rst	6
	.dw 	LCD

	pop	b



	lxi	h,msg11
	call	RPBMSG

	call	hit_any_key

	jmp	display_acknowledgements
	
	jmp	help_common_exit

;-----------------------------------------------------------------------
test_rom_mode:				; discover if REX is controlling main rom
					; set bit 5 if rex is in control
					; reset bit 5 if rex is not in control
	
	call	I_TESTROM
	call	GEN_1


	jnz	REX_no_control		; not zero, not equal, no control

REX_control:				; 0004-0006 = "enu" so REX is in control

	lda	SUFLAG
	ori	00100000b		; turn on bit 5
	jmp	test_rom_mode_done

REX_no_control:
	lda	SUFLAG
	ani	11011111b

test_rom_mode_done:
	sta	SUFLAG
	ret


;-----------------------------------------------------------------------
display_rom_mode:			; display a character at position 9
					; according to REX status
#if modeltype = 0			; M100
	lxi	h,0901h
#else					; T200
	lxi	h,0905h
#endif

	rst	6
	.dw	POSIT
	
	call	reverse_video

	lda	SUFLAG
	ani	00100000b		; test bit 5
	
	jnz	display_rom_mode1	; if not zero then REX is in control

					; REX is not in control'

	mvi	a,'M'
	jmp	display_rom_mode2
	
	
display_rom_mode1:
					; test to see what the status of REX is
	lda	ROMSEL			; get value of ROMSEL
	ani	01000000b		; get rom select indicator
	jz	display_primary

	mvi	a,'S'			; display an S
	jmp	display_rom_mode2
	
display_primary:
	mvi	a,'P'			; display a P

display_rom_mode2:
	rst	6
	.dw	LCD			; display reverse character

	call	normal_video
	ret

;-----------------------------------------------------------------------
blank_dir:
	mvi	b,160d			; 160 spaces
#if modeltype = 0			; M100
	lxi	h,0002h			; column 0 row 2
#else					; T200
	lxi	h,0006h			; column 0 row 2
#endif

	call	print_b_spaces_at_hl	; blank the screen
	ret

;-----------------------------------------------------------------------
sysmenu:				; special menu for system blocks
sysmenu_loop0:				; TYPE = 11 (system)

;sysmenu_loop0:				; restart of sysmenu
;sysmenu_loop2:				; use this to redraw screen	
;sysmenu_loop3:				; use this loop if cursor updated
;sysmenu_loop4:				; use this loop if entry invalid


	lxi	h,sysmenu
	shld	JMPVECT
					; sysblocks = 0, no system, sysbak

	lda	TYPE
	cpi	03h
	jnz	menu_loop0

;	call	clsr7

	call	read_active_block	; update active block info first
					; update ROMSEL

	call	display_rom_mode	; indicate if REX is in control of the ROM

	call	quick_scan_directory
	call	get_freeblock_status

	call	display_group_type




; set up the sort table, no sorting in this menu
	lxi	h,GEN_1+96
#if sysblocks = 0
	mvi	a,1h
	sta	COUNT			; Count = 4
	
	mvi	m,02h
	inx	h
	mvi	m,04h

#else
	mvi	a,05h
	sta	COUNT			; Count = 6

	mvi	m,00h
	inx	h
	mvi	m,01h
	inx	h
	mvi	m,02h
	inx	h
	mvi	m,03h
	inx	h
	mvi	m,04h
	inx	h
	mvi	m,05h
#endif
	

; sort table fixed, set to default

	call	blank_dir

	lda	2d
	sta	TEMP4

	lda	COUNT
	inr	a
	sta	COUNT
	
	call	display_directory_entries	; requires updated COUNT

	lda	COUNT
	dcr	a
	sta	COUNT

	xra	a
	sta	CURSOR




sysmenu_loop2:				; use this to redraw screen			
	
	call	clsr7
	call	clsr8

skey1:	call	RPRT00
#if modeltype = 0			; M100
	.db	08d,01d
#else					; T200
	.db	012d,01d
#endif

	.db	"Help Load "
	.db	00

skey3:	call	RPRT00
#if modeltype = 0			; M100
	.db	08d,11d
#else					; T200
	.db	012d,11d
#endif

	.db	"Use  "
	.db	00

skey6:	call	RPRT00		; display Cksm Exit
#if modeltype = 0			; M100
	.db	08d,31d
#else					; T200
	.db	012d,31d
#endif

	.db	"DeIn Exit"
	.db	00






sysmenu_loop3:			; use this loop if cursor updated

;	call	reverse_video
;	lda	CURSOR				; reverse video on new cursor
;	call	draw_cursor
;	call	normal_video

	xra	a		; now, find block number for item under cursor
	dcr	a
	sta	CURBLK		; store FF

	call	get_block_cursor

	push	h
	call	get_entry_date
	call	print_date

	pop	h
	call	get_entry_time
	call	print_time

	lda	CURBLK
	sta	FREBLK			; set FREBLK=CURBLK


					

	call	sysdisplay_text		; display text in row 6 describing selection

	lda	SUFLAG
	ani	00000100b		; test for zero in bit 2
	jz	sysmenu_loop4
					; if bit = 1 then show block info
	call	display_which_free_block
	call	display_current_block



sysmenu_loop4:				; use this loop if entry invalid
					; now get keyboard response, take action.

	call	draw_cursor

sysmenu_get_key:



	rst	6
	.dw	KYREAD		; wait for a character, returned in a
				; carry set - function key pressed
				; zero set - no key found



	jz	sysmenu_get_key
				; a key has been pressed 
				; arrow keys move cursor around
				; space advances cursor
				; enter "selects", action depends on cursor location
				; function keys take action depending on cursor location

	call	draw_cursor

				; check for function keys first

	jc	sysmenu_function_key_pressed		; if carry set then A holds fun key number
				; some other key pressed
				; up is 30, down is 31, left is 29, right is 28
				; space is 32, enter is 13
				; tab is 9


	rst	6
	.dw	UPCAS		; convert to uppercase

	cpi	'B'
	cz	toggle_showblock


	cpi	'I'		; display info about REX model etc.
	jz	show_REX_info

	cpi	'C'
	jz	display_checksum

	cpi	09d
	jz	increment_type

	call	process_cursor_type
; if CURSOR unchanged then menuloop4 else menuloop2

	lda	CURSOR
	cmp	h
	
	jnz	sysmenu_loop3		; loop here for cursor updates
	jmp	sysmenu_loop4		; loop 4 for no screen updates, bad entry


;-------------------------------------------------------------------
sysmenu_function_key_pressed:
; evaluate the function key action requested, based on mode.
; entry with a = function key number

	ora	a		; fn key 1
	jz	display_helpsys


	dcr	a		; fn key 2
	jz	Load_image_sysmenu	
	
	dcr	a		;3
	jz	Use_mainrom	
	dcr	a		;4
	dcr	a		;5
	dcr	a		; 6

	

	dcr	a		; 7
	jz	de_install_REX

	dcr	a		; 8
	jz	quit			; quit with a hooks table rebuld
					; HKFLAG controls hook table rebuild
					; unchanged at this point


	jmp	sysmenu_loop4		; must have pressed some other key, so loop back
					; no change to display
	


;----------------------------------------------------------------------------------

Use_mainrom:		; enter with CURBLK = rom to select

	call	clsr7
	lda	CURBLK
	cpi 	02d
	jz	use_continue

	cpi	04d
	jnz	sysmenu_loop2
	

use_continue:

	call	check_rex_control
	jz	sysmenu_loop2		; if REX not in control then don't switch
					; REX can switch the ROM
	call	check_is_rom_in_use
	jz	sysmenu_loop2		; if rom is already in use then don't switch to it

					; current rom is not in use
	lda	CURBLK
	cpi	02d
	jz	select_primary

	call	RPRT00
#if modeltype = 0			; M100
	.db	07h,01h
#else					; T200
	.db	0Bh,01h
#endif

	.db	"Switching to Secondary....."
	.db	00h
	jmp	select_continue

select_primary:
	call	RPRT00
#if modeltype = 0			; M100
	.db	07h,01h
#else					; T200
	.db	0Bh,01h
#endif

	.db	"Switching to Primary....."
	.db	00h

select_continue:	


	lxi	h,areyousure
	call	RPBMSG

	call	get_YN		; if "Y" return with zero set
				; if not "Y", return with zero reset

	jnz	sysmenu_loop2
				; ok, switch it.


	lda	CURBLK		; 2 or 4
	rrc			; 1 or 2
	dcr	a		; 0 or 1
	
	rrc
	rrc			; bit 6
	sta	ROMSEL		; change state of ROMSEL

	call	update_actblk_directory		; update actblk directory

; now we must update the current active optrom/active RAM record.
; we need to update the routines that manage this


	call	clsr7

	jmp	sysmenu
	

	
;----------------------------------------------------------------------------------
Load_image_sysmenu:			; CURBLK indicates target

					; if rex not in control, the load whichever
					; if rex is in control, then load the not selected one

	lda	SUFLAG
	ani	00100000b
	jz	load_either_rom		; if REX not in control then load either

					; REX is in control	
	call	check_is_rom_in_use
	jz	sysmenu_loop2			; restart menu1 if rom in use

					; current rom is in use so ok to erase and load

load_either_rom:			; enter here once target is validated



	lda	CURBLK
	cpi 	02d
	jz	load_primary

	call	RPRT00
#if modeltype = 0			; M100
	.db	07h,01h
#else					; T200
	.db	0Bh,01h
#endif

	.db	"Loading Secondary....."
	.db	00h
	jmp	load_continue

load_primary:
	call	RPRT00
#if modeltype = 0			; M100
	.db	07h,01h
#else					; T200
	.db	0Bh,01h
#endif

	.db	"Loading Primary....."
	.db	00h

load_continue:

	lxi	h,areyousure
	call	RPBMSG

	call	get_YN		; if "Y" return with zero set
				; if not "Y", return with zero reset

	jnz	sysmenu_loop2
				; ok, load it.

	call	check_flash_ready

; we will ignore directory entries at block 3 and 5 - in the table and always named as original

	lda	CURBLK

#if real_hardware=1
	call	erb0			; erase the block in a
#endif

; the old entry is now dead at 2 or 4
; blank out the entry

	lda	CURBLK			; reload pointer to entry
	call	find_directory_entryA	; point hl to this directory entry
	call	blank_directory_entry	; blank this directory entry (we erased the block)

; make a new entry with name "Blank!"

	call	load_make_entry
	lxi	d,Blank			; get pointer to 6 byte name		
					; hl holds new record location, de holds pointer to text
	call	write_directory_name	; write name bytes

; ready to load image
	jmp	Load_image_mrmmenu	; attempt to load a new image
					; if successful, new image name will be entered




Blank:	.db	"Blank!"
;----------------------------------------------------------------------------
get_block_cursor:		; find the block number under the cursor
				; exit with CURBLK updated and
				; hl pointing to directory entry


	
	lda	CURSOR		; cursor range 0-15
	mov	c,a
	mvi	b,00h

	lxi	h,GEN_1+96d	; point to 3rd table
	dad	b

	mov	a,m
	sta	CURBLK		; CURBLK updated with what is under CURSOR

	rlc			; a = a*2
	mov	c,a		; b = 00
	lxi	h,GEN_1+32d
	dad	b		; hl points to directory entry for CURSOR

	xchg
	lhlx			; hl = address of directory entry

	ret


;-------------------------------------------------------------------------------
sysdisplay_text:		; display test in row 6

	lda	CURBLK
	rlc			; 2x curblock = offset
	mov	c,a
	xra	a
	mov	b,a		; bc = offset
	lxi	h,systexttable
	dad	b		; get address in hl

	xchg
	lhlx			; get address for text string


	push	h
#if modeltype = 0			; M100
	lxi	h,0102h		; row 2 column 1
#else					; T200
	lxi	h,0106h		; row 6 column 1
#endif

	rst	6
	.dw	POSIT
	
	pop	h
	call	RPBUF

	ret
	
systexttable:
	.dw	sys0
	.dw	sys1
	.dw	sys2
	.dw	sys3
	.dw	sys4
	.dw	sys5


systext:
sys0:	.db	"REX software and directories       ",00h
sys1:	.db	"Backup of REX SYSTEM               ",00h
sys2:	.db	"Primary Main ROM 0000-7FFF         ",00h
sys3:	.db	"Primary Main ROM 8000-9FFF (T200)  ",00h
sys4:	.db	"Secondary Main ROM 0000-7FFF       ",00h
sys5:	.db	"Secondary Main ROM 8000-9FFF (T200)",00h




;----------------------------------------------------------------------------------
check_rex_control:		; return not zero if rex in control
				; return zero if rex in control

	lda	SUFLAG
	ani	00100000b
	rnz			; return if bit 5=1 - rex is in control

	lxi	h,msgmm1

	jmp	check_rom_common_exit


msgmm1:	.db	"REX not configured to provide main rom.",00h

;----------------------------------------------------------------------------------
check_is_rom_in_use:		
				; return zero if rom in use
				; return not zero if rom not in use
	lda	CURBLK		; 2 or 4
	rrc			; 1 or 2
	dcr	a		; 0 or 1
	rrc
	rrc			; place bit in bit 6 position	
	push	psw

	lda	ROMSEL
	ani	01000000b
	mov	b,a

	pop	psw

	cmp	b
	
	rnz				; return if not equal, switch ok

	lxi	h,msgmm2

check_rom_common_exit:
	call	RPBMSG


	call	DELAY

	call	clsmsg

	xra	a			; set zero bit
	ret
	
msgmm2:	.db	"Main ROM already selected!",00h



;-------------------------------------------------------------------
draw_cursor:

	push	h
	push	b
	push	d
	push 	psw

	lda	CURSOR
	mov	c,a

	call	display_get_position	; get position from count (reg c) for display
					; return row column in hl
	inr	h
	shld	CSRY		; store x and y cursor values l in y h in x

	mvi	b,07h
	rst	6
#if modeltype = 0
	.dw	059EDH		; entry to intercept cursor draw routine
#else
	.dw	06D75H
#endif
	pop	psw
	pop 	d
	pop	b
	pop	h

	ret





