;-----------------------------------------------------------------------
Load_image_common:			; load an image
					; deal correctly with the directory depending on flags


	call	check_free_block	; suflag used to indicate if there is a free block

Load_image_common1:	
	call	check_flash_ready

	lda	FREBLK
	call	check_blank
	
	lxi	h,Load_image_mainmenu1

	lda	TYPE
	cpi	03h
	jnz	Load_jmpvect

	lxi	h,Load_image_mrmmenu1
	
Load_jmpvect:
	shld	JMPVECT

	jmp	Load_image		; jump to Load_image	
	

;-----------------------------------------------------------------------
					; return path from general load
Load_image_mainmenu1:			; valid file name at F685
					; FREBLK is target

					; load status: 0=no load, 1= load in progress
					; commit status: 0=don't commit, 1=commit

					; states lc 00 10 11

					; if status 00
					; 	just return
					; if status 10
					; 	create new directory entry
					; 	mark new entry U
					; if status 11
					; 	create new directory entry
					; 	mark new entry A

	call	load_status
	jz	Load_image_done
					; if nothing loaded, done

					; load was started
					; start a new entry
	call	load_make_entry		; enter with FREBLK valid

					; hl points to free entry
	call	commit_status
	jz	Load_image_mainmenu_failed
					; there was a problem, mark U

	lxi	d,INPUT_BUFFER		; get pointer to 6 byte name		
	call	write_directory_name	; write name bytes


	jmp	Load_image_done		; done

Load_image_mainmenu_failed:		; mark entry as used not active

					; hl points to directory entry
	call	kill_directory_entry	; makes an A entry a U entry
					; entry hl = pointer to directory entry

Load_image_failed:
	call	load_clear
	
	lxi	h,menu_loop0
	shld	JMPVECT

	jmp	Load_failed_error	; display load failed error and back to menu

	
Load_image_done:
	call	load_clear
	jmp	menu_loop0


;-----------------------------------------------------------------------
					; return path from Load in SYS

Load_image_mrmmenu1:
					; valid file name at F685
					; FREBLK is target

					; load status: 0=no load, 1= load in progress
					; commit status: 0=don't commit, 1=commit

					; states lc 00 10 11

					; if status 00
					; 	just return
					; if status 10
					; 	erase FREBLK
					; if status 11 and model = T200
					; 	copy CURBLK adjacent to FREBLK adjacent
					;	erase CURBLK
					; 	copy FREBLK pair to CURBLK pair
					; 	create new directory entry for CURBLK
					; 	mark new entry A, with actual filename

					; if status 11 and model = M100
					;	erase CURBLK
					; 	copy FREBLK  to CURBLK 

					; 	create new directory entry for CURBLK
					; 	mark new entry A, with actual filename

	call	load_status
	jz	Load_image_done
					; if nothing loaded, done

					; load was started
	call	commit_status
					; hl points to free entry
	jz	Load_image_mrmmenu_failed
					; there was a problem, erase FREBLK

					; succesful
					; 	copy CURBLK adjacent to FREBLK adjacent
					;	erase CURBLK
					; 	copy FREBLK pair to CURBLK pair
					; 	create new directory entry for CURBLK
					; 	mark new entry A, with actual filename


	lda	CURBLK
	mov	b,a			; source
	lda	FREBLK
	mov	c,a			; target
	
	call	toggle_adjacent

	push	b
#if modeltype=1 	; T200
	call	copyblock_function	; copy from CURBLK' to FREBLK'
#endif


	lda	CURBLK
#if real_hardware=1
	call	erb0			; erase CURBLK
#endif

	pop	b
	mov	a,c
	mov	c,b
	mov	b,a			; reverse direction of copy	
	push	b
	call	copyblock_function		; copy from FREBLK' to CURBLK'

	pop	b
	call	toggle_adjacent
	
#if modeltype=1 	; T200
	call	copyblock_function	; copy from FREBLK to CURBLK
#endif

	lda	FREBLK
#if real_hardware=1
	call	erb0			; erase FREBLK
#endif

	lda	CURBLK			; reload pointer to entry
	sta	FREBLK			; set FREBLK to CURBLK for directory entry write
	call	find_directory_entryA	; point hl to this directory entry
	call	blank_directory_entry	; blank this directory entry (we erased the block)


	call	load_make_entry		; enter with FREBLK valid
					; exit hl points to new entry 

	lxi	d,INPUT_BUFFER		; get pointer to 6 byte name		
					; hl holds new record location, de holds pointer to text
	call	write_directory_name	; write name bytes
	

	jmp	Load_image_done		; done

Load_image_mrmmenu_failed:		; erase FREBLK

	lda	FREBLK
#if real_hardware=1
	call	erb0			; erase the block in a
#endif

	jmp	Load_image_failed	; display load failed error and back to menu


toggle_adjacent:
	push 	psw
	mov	a,b
	xri	00000001b
	mov	b,a
	mov	a,c
	xri	00000001b
	mov	c,a
	pop	psw
	ret
		
;-----------------------------------------------------------------------

load_make_entry:			; enter with FREBLK valid
					; exit hl points to new entry

	lda	TYPE
	sta	LASTB

	mvi	a,0FFH
	call	find_directory_entry	; get free directory entry

	push	h			; hl points to free entry
	lda	FREBLK
	call	write_directory_block	; start the directory entry

	pop	h			; load was successful
	push	h			; store pointer to entry on stack
	call	write_directory_lastb

	call	get_current_time
	pop	h			; hl points to new entry
	push	h
	call	store_entry_time

	call	get_current_date
	pop	h
	push	h			; hl points to new entry
	call	store_entry_date

	pop	h

	ret


;-----------------------------------------------------------------------
load_clear:
	call	clear_commit
	call	clear_load
	jmp	fix_input_buffer	
;-----------------------------------------------------------------------
; Load_image
; based on Wilson Van Alst's R2X100
; uses TPDD protocol to load 32k binary files.
; entry
; 	FREBLK holds target block
;
; get a filename, and load the image into FREBLK
; return with status set
					; load status: 0=no load, 1= load in progress
					; commit status: 0=don't commit, 1=commit

					; states lc 00 10 11
;-----------------------------------------------------------------------


Load_image:				; JMPVECT must be set - where it returns to
					; FREBLK must be set.

	call	setup_TPDD


; set up the filename
	call	name_setup_load		; filename stored at F685
	jz	common_exit		; if zero, exit, no need to set up stack

; display loading filename

	call	wait_TPDD_ready
	jz	common_exit

	call	clsr7
	
	lxi	h,msgt2
	call	print_tpddaction


	call	I_WRITE128		; code to quickly write 128 bytes

	call	TPDD_prep_open
										
	lxi	h,INPUT_BUFFER
	mov	a,m
	sta	TEMP3			; store first character of name
	mvi	a,03h
	mov	m,a			; place mode here

	LXI 	H,0101H			; load HL with 0101		
	CALL	SEND_TPDD_COMMAND	; call - send command	

	lxi	h,INPUT_BUFFER
	lda	TEMP3
	mov	m,a			; restore first character of name			
	
	call	set_load		; set load pending flag
					; now if there is an error, the block get's marked bad

					; now into loops for reading blocks of data, and poking them into REX
get_128_bytes:				; stack = pointer to memory

	LXI 	H, 0003H		; load HL with 0003 read file
	CALL	SEND_TPDD_COMMAND	; call - send command	
					; at this point, file data has been moved to F6B7
					; F6B7 is a buffer
	xchg				; de = start of payload data

	lda	FREBLK			; load a with available block
	mov	c,a			; store in C

	lhld	TEMP2			; load hl with TEMP2 pointer

	call	GEN_1			; write 128 bytes	

	shld	TEMP2	
								
	mov	a,h
	ani	00000011b
	ora	l
	jnz	load_1

#if modeltype = 0
	mvi	a,0FFh
	out	0FFh		; send to LCD
#else
	call	TBAR		; uses PSW
#endif


load_1:

	lhld	TEMP2
	MOV 	A,H		
	RAL				;rotate it left
	JNC	get_128_bytes		;if carry bit not set then loop back - get another 80 bytes

	lxi	h,msgt4

	jmp	TPDD_finish



;-----------------------------------------------------------------------
setup_TPDD:
	call	set_DTR
	call	set_parameters

	call	clear_commit		; clear the commit status
	jmp	clear_load		; clear the load status
;-----------------------------------------------------------------------

print_tpddaction:
	call	RPBMSG
	
	mvi	d,09h			; print 9 characters
	lxi	h,INPUT_BUFFER		; point to input buffer
	call	print_d			; print out name

	di	
	call	I_BARS
	call	GEN_1			; set up bar indicator
	ret
;-----------------------------------------------------------------------
TPDD_prep_open:
	LXI 	H,0000h				
	shld	TEMP2			; store rom pointer	


					;filename is prepped for use to "open file"								
	LXI 	H,0007H			;load HL with 0007 drive status					
	CALL	SEND_TPDD_COMMAND	;call - send command

	LXI 	H,1A00H			;load HL with 1A00 find file					
	CALL	SEND_TPDD_COMMAND	;call - send command
	ret
;-----------------------------------------------------------------------
TPDD_finish:
	ei
	call	RPBMSG
				
	LXI 	H,0002H			;load HL with 0002 close file				
	CALL	SEND_TPDD_COMMAND	;call - send command								

	call	set_commit		; indicate success on image load

	jmp	common_exit		; return to call location
					; no need to reset the stack
;-----------------------------------------------------------------------	

; Save_image
; based on Wilson Van Alst's R2X100
; uses TPDD protocol to load 32k binary files.
;-----------------------------------------------------------------------

Save_image:

	call	setup_TPDD

	call	name_setup_save
	jz	common_exit		; if zero, exit, no need to set up stack

	call	wait_TPDD_ready
	jz	common_exit

	lxi	h,msgt3
	call 	print_tpddaction


	call	I_READ128		; read 128 bytes from flash

	call	TPDD_prep_open

									
	LXI 	H,0101H			;load HL with 0101	
	MVI 	A,01H			;load A with 01	(mode)	
	STA	INPUT_BUFFER		;store 01 in F685	
	CALL	SEND_TPDD_COMMAND	;call - send command				


send_128_bytes:	
	lhld 	TEMP2			; h points to block data

	lda	CURBLK			; load a with current block
	mov	c,a			; store in C

	lxi	d,INPUT_BUFFER
					; build command data with 128 bytes of block data
	call	GEN_1			; read 128 bytes

	shld	TEMP2			; store counter in TEMP2

	LXI 	H, 8004H		;load HL with 8004 send 128 bytes of file
	CALL	SEND_TPDD_COMMAND	;call - send command	

	lhld	TEMP2
	mov	a,h
	ani	00000011b
	ora	l
	jnz	send_1

#if modeltype = 0
	mvi	a,0FFh
	out	0FFh		; send to LCD
#else
	call	TBAR		; uses PSW
#endif


send_1:
	lhld	TEMP2
	MOV 	A,H		
	RAL				;rotate it left
	JNC	send_128_bytes		;if carry bit not set then loop back - send another 80 bytes


	lxi	h,msgt5
	call	TPDD_finish				
												
;-------------------------------------------------------------------------------												
;This routine sends the command, and loads the response from disk							
;-------------------------------------------------------------------------------
												
SEND_TPDD_COMMAND:	

	di
					;input HL = first 2 bytes after ZZ  - Z Z L H
					;send out command bytes, stored in input buffer at F685
					;wait for response
					;place all response bytes into buffer
					;returns with
					;load b with first response byte
					;load a with 3rd response byte
					;zero bit zet if a=0, reset if a <>0												
	mvi	c,00h			;zero out C
					;these bytes hold the first 2 bytes following ZZ
	LXI 	D,INPUT_BUFFER-2d	;load DE with F683
	SHLX				;store HL at F683=L F684=H (LH 0700 ,001A, 0101, 0300, 0200)
					;command data stored at beginning of command text
	INR 	H			;increment H								
	INR 	H			;increment H; add 2 bytes to loop counter
					;space for "F" and directory reference
					;h holds the number of command bytes (h = length not including ZZ)								
	MVI 	A, 5AH			;load A with 5A "Z"
	call	send_a_byte
	call	send_a_byte		;this routine preserves hl

snd_com_loop:													
	LDAX 	D			;load A from contents of DE	
	call	send_a_byte
												
	ADD 	C			;add C to A		a=contents of memory at DE - add A to current checksum							
	MOV 	C,A			;put in C		update checksum							
	INX 	D			;increment DE		move to next byte							
	DCR 	H			;decrement h - h holds the number of characters							
	JNZ	snd_com_loop		;loop back until h characters have been sent									

	xchg				; de points to first response byte location in buffer
	lxi	b,0020h
	dad	b			; add space for name in command location
	shld	TEMP1			; TEMP1 holds next byte location
	xchg

	inx	d			; de points to second incoming byte	


	CMA				; complement A - A holds the checksum
	call	send_a_byte													


; start receive function here

; checksum of the response is the ones compliment of the lsb of the sum
; of all bytes excluding the checksum

; sum them up, subtract the checksum (last byte), xor FF  (ones compliment)  compare
; store checksum at the "next byte" location

					; command is sentnow get response
	lxi	h,01000h
	shld	TEMP5			; TEMP5 holds countdown timer

	lhld	TEMP1
	xra	a
	mov	m,a			; store 00 at TEMP1 (starting checksum)

get_response_loop:			; wait in this loop till response is read in
	rim				; works in M100 and T200!
	ani	00100000b
	jnz	get_serial_byte		; if byte present, get it!

	mvi	a,020h			; add some delay
response_delay:
	dcr	a
	jnz	response_delay
	
	lhld	TEMP5	
	dcx	h			; countdown....
	shld	TEMP5

	mov	a,h
	ora	l

	jz	TPDD_response_error	; countdown timer for lack of response.

	jmp	get_response_loop

get_serial_byte:
	lxi	h,01000h
	shld	TEMP5			; reset counter
	xra	a
	
#if modeltype=0			; M100
	in	0c8h
#else				; T200
	in 	0CEh
#endif


	lhld	TEMP1			; get location for next receive character
					; should hold checksum

	mov	c,m			; get checksum
	mov	m,a			; store data
	add	c			; add to checksum

	inx	h			; hl points to next character location
	mov	m,a			; store new checksum
	shld	TEMP1			; store new next character


	ldax	d			; get byte from de location
					; hl holds next byte location
	inr	a
	inr	a			; add 2 to a
	mov	c,a
	mvi	b,00h		
	dsub				; subtract length bytes from location		
					; end when result = de

	mov	a,h
	cmp	d
	
	jnz	get_response_loop

	mov	a,l
	cmp	e
	
	jnz	get_response_loop

;  1   2   3   4   5
; XX  02  YY  YY  CS
; a1  a2  a3  a4  a5  HL
;     DE

; now compare the checksums

	lhld	TEMP1
	mov 	a,m			; load a with checksum
	dcx	h
	sub	m			; subtract response checksum value (not included in total)
					; a holds actual checksum
	xri	0FFH			; toggle all bits
	cmp	m			; compare to response checksum at m
	jnz	response_checksum_error


	xchg
	dcx	h			;get back original start of response

	MOV 	B,M			;load B from M load B with first received character
	INX 	H			;increment HL				
	INX 	H			;increment HL skip ahead two bytes
	xra	a
	ADD 	M			;add M to A load A with M (A was zero)
					;return with zero bit set if M=0

					; check for an error response
					; hl points to start of payload of response

			



;----------------------------------------------------------------------
disk_error:			; disk error display
				; if reponse byte is 12 then an error is possible
	mov	c,a
	push	b		; temp store a
	mov	a,b
	sui	12h		; see if b=12h
	pop	b		
	mov	a,c		; restore a
	rnz			; return if not an possible error response

				; there was a disk error
				; if a = 0 then return - no error
	ora	a		; check if a = 0
	rz			; return from read response routine if no error


				; there was an error!

	ani	11110000b		; and with F0
	rrc				; error in range 10-80
	rrc
	rrc				; rotate right 3 times, we want 2x 
	
	lxi	h, error_table
	mov	c,a
	mvi	b,00h
	dad	b			; offset hl with error reference hl = hl + 2*a
					; hl points to error message
	xchg				; place pointer in de
	lhlx				; hl = (DE)
					; hl points to error message	
	
	jmp	common_error		; there was a disk error, so bail out!



error_table:
	.dw	unspecified_error
	.dw	file_name_error
	.dw	unknown_error
	.dw	command_error
	.dw 	read_error
	.dw	write_protect
	.dw	disk_full
	.dw	insert_disk
	.dw	hardware_error


errors:
unspecified_error:
	.db	"Unspecified error",00h		; code 0xh
file_name_error:
	.db	"File name error",00h		; code 1xh
unknown_error:
	.db	"Unknown error",00h		; code 2xh
command_error:
	.db	"Command error",00h		; code 3xh
read_error:
	.db	"Read error",00h		; code 4xh
write_protect:
	.db	"Write protected",00h		; code 5xh
disk_full:
	.db	"Disk full",00h			; code 6xh
insert_disk:
	.db	"Insert disk",00h		; code 7xh
hardware_error:
	.db	"Hardware error",00h		; code 8xh

;00 - normal (no error)
;10 - file does not exist 11 - file exists
;30 - no filename 31 - dir search error 35 - bank error
;36 - parameter error 37 - open format mismatch 3f - end of file
;40 - no start mark 41 - crc check error in ID 42 - sector length error
;44 - format verify error 46 - format interruption 47 - erase offset error
;49 - crc check error in data 4a - sector number error 4b - read data timeout
;4d - sector number error 50 - disk write protect 5e - un-initilized disk
;60 - directory full 61 - disk full 6e - file too long
;70 - no disk 71 - disk change error


;----------------------------------------------------------------------
Load_failed_error:			; error when load does not succeed
	lxi	h,msgt10
	jmp	common_error
;----------------------------------------------------------------------
TPDD_response_error:			; error when TPDD does not respond
	lxi	h,msgt11
	jmp	common_error
;----------------------------------------------------------------------				
send_error:				; print error when esc hit
	lxi	h,msgt9
	jmp	common_error
;----------------------------------------------------------------------				
response_checksum_error:		; print error when checksum bad
	lxi	h,msgt7
	jmp	common_error
;----------------------------------------------------------------------				
esc_abort:				; print error when esc hit
	lxi	h,msgt6
	jmp	common_error


;----------------------------------------------------------------------
send_a_byte:

#if modeltype = 0		; M100
					; byte in a
	push	psw
	push	h

	lxi	h,0000h		; set countdown limit, error if wait too long
send_byte_wait_buffer:	
	dcx	h
	mov	a,h
	ora	l
	jz	send_error
	IN	0D8H
	ANI	00010000b			; bit 4 is transmit buffer ready
	JZ	send_byte_wait_buffer		; wait for buffer to clear  TBRE = 1
	
	pop	h
	pop	psw
	OUT	0C8H
	ret


#else				; T200

					; byte in a
	push	psw
	push	h

	lxi	h,0000h		; set countdown limit, error if wait too long
send_byte_wait_buffer:	
	dcx	h
	mov	a,h
	ora	l
	jz	send_error
	IN	0CFH
	ANI	00000001b			; bit 1 is transmit buffer ready
	JZ	send_byte_wait_buffer		; wait for buffer to clear  TBRE = 1
	
	pop	h
	pop	psw
	OUT	0CEh				; write data byte
	ret

#endif



;---------------------------------------------------------------------
;set_up_filename:			; get 6 byte name at input buffer
					; format it for command
;	lda	CURSOR
;	mov	b,a
;	lda	COUNT
;	cmp	b
;	jnz	name_setup_save

name_setup_load:
	lxi	h,msgt1
	call	RPBMSG

#if modeltype = 0 		; M100
	lxi	h,01E08h
#else				; T200
	lxi	h,01E0Ch	
#endif	

	call	get_6_byte_name
	rz
	
	jmp	name_setup_continue

name_setup_save:
	lda	CURBLK
	call	find_directory_entryA

	inx	h			; hl points to name text
	lxi	d,INPUT_BUFFER		; point de to input buffer

	mvi	b,06d

set_up_filename1:
	mov	a,m

	push	b
	push	h
	push	d
	rst	6
	.dw	UPCAS		; ensure it is upper case
	pop	d
	pop	h
	pop	b

	stax	d
	inx	h
	inx	d
	dcr	b
	jnz	set_up_filename1
	

name_setup_continue:			; image name copied to input buffer

	lxi	h,INPUT_BUFFER+6d
	mvi	b,18d
	call	add_whitespace
					; adding whitespace to the end of the text
	lxi	d, 0046H		; load d with 0046
	xchg				; exchange with h - h was the end of the text buffer
	shlx				; store 0046 at the end of the text buffer


			
	lxi 	h,INPUT_BUFFER+6d	; load hl with F68B
	mvi 	m,2EH			; store 2E at F68B "."
	inx 	h			; increment HL
	push	h			; temp store

	lxi	h,extensions		; point hl to extensions
	lda	TYPE
	rlc				; times 2
	mov	e,a
	xra	a
	mov	d,a			; de = type*2

	dad	d			; hl points to extension text

	xchg
	lhlx				; load hl with 2 bytes from extension text
	pop	d			; restore d
	shlx				; store extension at de

					; filename is prepped for use
					; return codes 00 = success	01,02, 03 errors

	ori	01h
	ret				; return with zero bit reset

;---------------------------------------------------------------------------------
set_load:				; use bit6 of SUFLAG to indicate load pending.
	lda	SUFLAG			; set the load status
	ori	01000000b		; set bit 6
	sta	SUFLAG
	ret
;---------------------------------------------------------------------------------
clear_load:				; use bit6 of SUFLAG to indicate load pending.
	lda	SUFLAG			; clear the load status
	ani	10111111b		; clear bit 6
	sta	SUFLAG
	ret
;---------------------------------------------------------------------------------
load_status:				; use bit6 of SUFLAG to indicate load pending.
	lda	SUFLAG			; notzero = load pending
	ani	01000000b		; zero = no load pending		
	ret
;---------------------------------------------------------------------------------
set_commit:				; confirm commit to memory, process successful
	lda	SUFLAG
	ori	10000000b		; set bit 6
	sta	SUFLAG
	ret
;---------------------------------------------------------------------------------
clear_commit:				; use MSB of SUFLAG to control status of load.
	lda	SUFLAG			; clear the commit status
	ani	01111111b
	sta	SUFLAG
	ret
;---------------------------------------------------------------------------------
commit_status:				; return status of commit
	lda	SUFLAG			; zero = load failed
	ani	10000000b		; notzero = load success
	ret


;---------------------------------------------------------------------
set_parameters:
	mvi	a,0FFh
	adi	01h			; set carry for RS232
	LXI 	H,SERPAR		; POINT HL TO SERIAL PARAMETERS	
	rst	6								
	.dw	SETSER			; Set RS232 parameters from string at M
	ret

;------------------------------------------------------------------
set_DTR:

#if modeltype=0 
	in 	0BAh
	ani	10111111b		; assert DTR,
	ori	10000000b		; de assert RTS
	out	0BAh			; bit 7 = RTS, bit 6 = DTR
	ret				; IGNORE DSR
#else
	LDA 	0FCFEH
	ori	00100111b
	STA 	0FCFEH
	OUT 	0CFH
	ret


#endif

;---------------------------------------------------------------------------------
wait_TPDD_ready:			; press any key when TPDD ready
					; escape returns to menu

	lxi	h,msgt8
	call	RPBMSG
	
	call	hit_any_key
	jz	common_exit_stack	; if escape then terminate

	ret				; zero set if escape


;---------------------------------------------------------------------------------
extensions:				; 00=ramimage, 01=romimage, 10=bootimage
ext1:	.db	"BY"			; ram extension, type 00
ext2:	.db	"BX"			; rom extension, type 01
ext3:	.db	"BZ"			; OS extension, type 10
ext4:	.db	"BR"			; REX extension, type 11 (SYS)

;---------------------------------------------------------------------------------
msgt1:	.db	"Loading from image file name: ",00h
msgt2:	.db	"Loading 0:",00h
msgt3:	.db	"Saving 0:",00h
msgt4:	.db	"Load image complete!",00h
msgt5:	.db	"Save image complete!",00h
msgt6:	.db	"ESC key pressed...aborting.",00h
msgt7:	.db	"Checksum error...aborting.",00h
msgt8:	.db	"Hit any key when TPDD ready.",00h
msgt9:	.db	"Error sending command!",00h
msgt10:	.db	"Load file from TPDD failed!",00h
msgt11:	.db	"No response from TPDD!",00h

